/******************************************************************************\
**
**     COMPANY:	LDW Software
**
**      MODULE:	WorldView
**
** DESCRIPTION:	Maintains the world view window
**
**      AUTHOR:	Adrian Francis
**
\******************************************************************************/

#ifndef WORLDVIEW_H
#define WORLDVIEW_H

/******************************************************************************\
** LIBRARY INCLUDES
\******************************************************************************/
#include "ldwTypes.h"

/******************************************************************************\
** CONSTANT DEFINITIONS
\******************************************************************************/
const int WorldView_c_iLeft	= 205;
const int WorldView_c_iRight = 795;
const int WorldView_c_iTop = 5;
const int WorldView_c_iBottom = 480;

const int WorldView_c_iWidth = WorldView_c_iRight - WorldView_c_iLeft;
const int WorldView_c_iHeight = WorldView_c_iBottom - WorldView_c_iTop;

const int WorldView_c_iLeftMargin = WorldView_c_iLeft + 40;
const int WorldView_c_iRightMargin = WorldView_c_iRight - 60;
const int WorldView_c_iTopMargin = WorldView_c_iTop + 40;
const int WorldView_c_iBottomMargin = WorldView_c_iBottom - 100;
const int WorldView_c_iScrollSpeed = 1;

/******************************************************************************\
** CLASS DEFINITIONS
\******************************************************************************/
class CWorldView
{
private:

	//class data
	ldwPoint				m_sTopLeft;

	bool					m_bIsFastScroll;
	ldwPoint				m_sScrollMod;

	static const ldwRect	scm_sRect;

	//class methods
	//setup
	public:		CWorldView(void);
	public:		void Reset();

	// process
	public:		void KeepInView(const ldwPoint sPos);
	public:		void Scroll(const ldwPoint sVelocity);
	public:		void ScrollTo(const ldwPoint sPos);
	public:		void ScrollAmount(const ldwPoint sOffset);
	public:		void StopScrolling();
	public:		void Update();
	public:		void CenterAtWorldPos(const int iXPos, const int iYPos);
	private:	void Constrain();

	// access
	public:		void SetPos(const ldwPoint sPos) { m_sTopLeft = sPos; Constrain(); StopScrolling(); }
	public:		void SetPos(const int iXPos, const int iYPos) { m_sTopLeft.x = iXPos; m_sTopLeft.y = iYPos; Constrain(); StopScrolling(); }
	public:		const ldwPoint Pos() const { return m_sTopLeft; }
	public:		const ldwPoint WorldPosAtCenter() const { return ldwPoint(m_sTopLeft.x + WorldView_c_iLeft + (WorldView_c_iWidth / 2), m_sTopLeft.y + WorldView_c_iTop + (WorldView_c_iHeight / 2)); }
	public:		const ldwPoint RandomPoint() const;
	public:		const ldwRect Rect() const { return scm_sRect; }
	public:		const bool IsScrolling() const { return m_sScrollMod.x != 0 || m_sScrollMod.y != 0; }
	public:		const bool IsFastScrolling() const { return m_bIsFastScroll; }
	public:		const bool IsInView(const ldwPoint sPos) const { return sPos.x >= scm_sRect.left + m_sTopLeft.x && sPos.x <= scm_sRect.right +m_sTopLeft.x && sPos.y >= scm_sRect.top + m_sTopLeft.y && sPos.y <= scm_sRect.bottom + m_sTopLeft.y; }
};

/******************************************************************************\
** GLOBAL DECLARATIONS
\******************************************************************************/
extern CWorldView		WorldView;

#endif

/******************************************************************************\
** END OF FILE
\******************************************************************************/